#ifndef AHLGREN_GRAPH
#define AHLGREN_GRAPH

#include <vector>
#include <list>
#include <deque>
#include <map>
#include <algorithm>
#include <unordered_map>
#include <regex>

#include "term.h"
#include "mode.h"
#include "bottom.h"
#include "parameters.h"
//#include "propositional.h"

namespace lp {


	template <typename Sol>
	class lattice {
	public:
		typedef Sol sol_type;
		struct node {
			node() : rank(0) {}
			node(node&& n) : visit(std::move(n.visit)), rank(n.rank), sol(std::move(n.sol)), 
				cand(std::move(n.cand)), up(std::move(n.up)), down(std::move(n.down)) {}
			//node& operator=(node&);
			// Node ID is defined as the first visit index
			int id() const { return visit.front(); }
			std::vector<int> visit;
			int rank;
			sol_type sol;
			Functor cand;
			std::deque<node*> up,down;
		private:
			node(const node&);
			node& operator=(const node&);
		};
		typedef std::multimap<int,node> layer_type;
		typedef typename layer_type::iterator iterator;
		typedef typename layer_type::const_iterator const_iterator;
		typedef typename layer_type::size_type size_type;
		typedef typename layer_type::value_type value_type;

		// Constructors
		lattice() : visitc(0) {}
		lattice(const std::string& s) : visitc(0), na(s) {}
		lattice(std::string&& s) : visitc(0), na(std::move(s)) {}
		lattice(const std::string& s, const Functor& cl) : visitc(0), na(s), bot(copy(cl)), bot_size(bot->body_size()) {}
		lattice(std::string&& s, const Functor& cl) : visitc(0), na(std::move(s)), bot(copy(cl)), bot_size(bot->body_size()) {}
		lattice(const std::string& s, term*& cl) : visitc(0), na(s), bot(std::move(cl)), bot_size(bot->body_size()) {}
		lattice(std::string&& s, term*& cl) : visitc(0), na(std::move(s)), bot(std::move(cl)), bot_size(bot->body_size()) {}
		// Move
		lattice(lattice&& l) : visitc(l.visic), na(std::move(l.name)), layers(std::move(l.layers)) {}
		lattice& operator=(lattice&& l);
		// Clear
		void clear() { visitc = 0; na.clear(); layers.clear(); bot.clear(); bot_size = 0; }
		// Name
		const std::string& name() const { return na; }
		void name(std::string&& s) { na = std::move(s); }
		void name(const std::string& s) { na = s; }
		// Bottom, example
		void bottom(const Functor& cl, const Functor& e) { bot = cl; bot_size = bot.body_size(); ex = e; }
		const Functor& seed() const { return ex; }
		const Functor& bottom() const { return bot; }
		// Size
		size_type size() const { return layers.size(); }
		bool empty() const { return layers.empty(); }
		// Insert solution
		bool insert(const sol_type&, const Functor& cand);
		bool insert(sol_type&&, const Functor& cand);

		// Find solution
		const_iterator find(const sol_type&) const;
		iterator find(const sol_type&);

		// Iterators
		iterator begin() { return layers.begin(); }
		iterator end() { return layers.end(); }
		const_iterator begin() const { return layers.begin(); }
		const_iterator end() const { return layers.end(); }
		// Ranges
		std::pair<iterator,iterator> equal_range(const value_type& p) { return layers.equal_range(p); }
		std::pair<const_iterator,const_iterator> equal_range(const value_type& p) const { return layers.equal_range(p); }

		struct hashf {
			// Hash solutions based on mask
			size_t operator()(const sol_type& s) const {
				long long n = 0;
				long long f = 1;
				for (auto i = s.mask().begin(); i != s.mask().end(); ++i) {
					n += f * (*i);
					f *= 2;
				}
				return static_cast<size_t>(n);
			}
		};
		// Set node ranks: we input bsf because this is not always possible to get from fitnesses (see MinSet)
		void compute_rank(int noise, int inflate);

		// Print lattice
		std::string node_id(int) const;
		std::string format(const Functor&) const;
		void print_subgraph(ostream& os, const parameters&, const vector<Mode>*) const;
		void print_node(ostream&, const node&, const parameters&, const vector<Mode>*) const;

	protected:
		std::string na; // name
		Functor bot; // bottom clause
		int bot_size; // bottom size
		Functor ex; // seed example
		layer_type layers;
		int visitc; // we need to keep track of visits separately in case of revisits
		// Member functions
		void attach_edges(iterator at);
	};


	template <typename Cand>
	lattice<Cand>& lattice<Cand>::operator=(lattice&& l)
	{
		if (this != &l) {
			visitc = l.visic;
			na = std::move(l.na);
			layers = std::move(l.layers);
		}
		return *this;
	}


	template <typename Cand>
	void lattice<Cand>::attach_edges(iterator at)
	{
		// Build pointers to/from all candidates one level above
		// Look one level above
		const bitstring& cand = at->second.sol.mask();
		auto range = layers.equal_range(at->first - 1);
		std::for_each(range.first,range.second,[&](layer_type::value_type& p){
			if (is_subset(p.second.sol.mask(),cand)) {
				p.second.down.push_back(&at->second);
				at->second.up.push_back(&p.second);
			}
		});
		// Look one level below
		range = layers.equal_range(at->first + 1);
		std::for_each(range.first,range.second,[&](layer_type::value_type& p){
			if (is_subset(cand,p.second.sol.mask())) {
				at->second.down.push_back(&p.second);
				p.second.up.push_back(&at->second);
			}
		});
	}

	template <typename Cand>
	bool lattice<Cand>::insert(const sol_type& c, const Functor& can)
	{
		++visitc; // register another visit, whether it's a dupe or not
		// Check for duplicate
		const auto level = std::count(c.mask().begin(),c.mask().end(),1);
		auto range = layers.equal_range(level);
		auto dupe = std::find_if(range.first,range.second,[&](const layer_type::value_type& p){
			return p.second.sol == c;
		});
		if (dupe != range.second) {
			// Register another visit
			dupe->second.visit.push_back(visitc);
			return false;
		}
		// New: insert
		node gnode;
		gnode.sol = c;
		gnode.cand = can;
		gnode.visit.push_back(visitc);
		auto at = layers.insert(layer_type::value_type(level,std::move(gnode)));
		attach_edges(at);
		return true;
	}

	template <typename Cand>
	bool lattice<Cand>::insert(sol_type&& c, const Functor& can)
	{
		++visitc; // register another visit, whether it's a dupe or not
		// Check for duplicate
		const auto level = c.candidate()->body_size();
		auto range = layers.equal_range(level);
		auto dupe = std::find_if(range.first,range.second,[&](const layer_type::value_type& p){
			return p.second.sol == c;
		});
		if (dupe != range.second) {
			// Register another visit
			dupe->second.visit.push_back(visitc);
			return false;
		}
		// New: insert
		node gnode;
		gnode.sol = std::move(c);
		gnode.cand = can;
		gnode.visit.push_back(visitc);
		auto at = layers.insert(layer_type::value_type(level,std::move(gnode)));
		attach_edges(at);
		return true;
	}

	template <typename Cand>
	auto lattice<Cand>::find(const sol_type& s) const -> const_iterator
	{
		// Only search on correct level
		auto range = layers.equal_range(s.candidate()->body_size());
		for ( ; range.first != range.second; ++range.first) {
			if (range.first->second.sol == s) return range.first;
		}
		return end();
	}

	template <typename Cand>
	auto lattice<Cand>::find(const sol_type& s) -> iterator
	{
		// Only search on correct level
		auto range = layers.equal_range(s.candidate()->body_size());
		for ( ; range.first != range.second; ++range.first) {
			if (range.first->second.sol == s) return range.first;
		}
		return end();
	}

	template <typename Cand>
	void lattice<Cand>::compute_rank(int noise, int inflate)
	{
		std::vector<node*> rankv,deflated;
		rankv.reserve(size());
		deflated.reserve(size());
		for (auto& p : *this) {
			const auto& sol = p.second.sol;
			// Only rank potential candidates
			if (sol.is_valid() && sol.is_consistent(noise)) {
				if (100.0 * std::count(sol.mask().begin(),sol.mask().end(),1) / sol.pos() < inflate) {
					rankv.push_back(&p.second);
				} else {
					deflated.push_back(&p.second);
				}
			} else {
				//std::cerr << "Not ranking candidate: "; 
				//print_candidate(std::cerr, p.second.cand); 
				//std::cerr << "\n";
				//std::cerr << "valid: " << sol.is_valid() << "\n";
				//std::cerr << "consistent: " << sol.is_consistent(noise) << "\n";
				//std::cerr << "inflate: " << 100.0 * std::count(sol.mask().begin(),sol.mask().end(),1) / sol.pos() << " < " << inflate << "\n";
			}
		}
		// Sort first highest fitness first
		std::sort(rankv.begin(),rankv.end(),[&noise](const node* s, const node* t){ return t->sol.fcompare(s->sol,noise) < 0; });
		std::sort(deflated.begin(),deflated.end(),[&noise](const node* s, const node* t){ return t->sol.fcompare(s->sol,noise) < 0; });
		// Set ranks
		int r = 0;
		for (auto p : rankv) p->rank = ++r;
		for (auto p : deflated) p->rank = -(++r);
	}

	template <typename Cand>
	std::string lattice<Cand>::format(const Functor& c) const
	{
		std::stringstream ss;
		print_candidate(ss,c);
		std::string s = ss.str();
		s = std::regex_replace(s,std::regex("<"),std::string("&lt;"));
		s = std::regex_replace(s,std::regex(">"),std::string("&gt;"));
		return s;
	}

	template <typename Cand>
	std::string lattice<Cand>::node_id(int k) const
	{
		return "node_" + name() + "_" + std::to_string(long long(k));
	}

	template <typename Cand>
	void lattice<Cand>::print_node(
		ostream& os, 
		const node& nd, 
		const parameters& params,
		const vector<Mode>* modes) const
	{
		// Print node IDs
		os << "\t\t" << node_id(nd.visit.front()); // use first visit as id
		os << " [shape=none,label=< <TABLE><TR>";
		// Print index
		os << "<TD>" << nd.visit.front();
		std::for_each(++nd.visit.begin(),nd.visit.end(),[&](int v){ os << "," << v; });
		os << "</TD>";
		// Print fitness
		os << "<TD>";
		if (nd.sol.fitness() == worst_fitness
			|| nd.sol.fitness() == invalid_fitness) {
				os << "&#8722;&infin;";
		} else if (nd.sol.fitness() == numeric_limits<double>::max() 
			|| nd.sol.fitness() == numeric_limits<double>::infinity()) {
				os << "&infin;";
		} else {
			os << nd.sol.fitness();
		}
		os << "</TD>";
		// Determine if clause is: invalid, inconsistent, consistent
		enum cand_state { INVALID, INCONSISTENT, CONSISTENT };
		cand_state cstate;
		//std::cerr << "print_Node, calling make_io_map on: " << nd.cand << "\n";
		const auto iom = make_io_map(nd.cand,*modes);
		if (is_mode_conformant(nd.cand,params,iom)) {
			cstate = (nd.sol.is_consistent(params.force_int(parameters::noise)) ? CONSISTENT : INCONSISTENT);
		} else {
			cstate = INVALID;
		}
		// Print positive coverage
		os << "<TD";
		if (cstate == CONSISTENT) os << " bgcolor=\"green\"";
		os << ">" << nd.sol.pos() << "</TD>";
		// Print negative coverage
		os << "<TD";
		if (cstate == INCONSISTENT) os << " bgcolor=\"red\"";
		os << ">" << nd.sol.neg() << "</TD>";
		// Print length
		const int body_size = std::count(nd.sol.mask().begin(),nd.sol.mask().end(),1);
		os << "<TD>" << body_size << "</TD>";
		os << "</TR>";
		// Second line --->
		// Print rank
		os << "<TR><TD";
		// Highlight best solution
		if (nd.rank == 1) os << " bgcolor=\"blue\"";
		else if (nd.rank > 0) os << " bgcolor=\"green\"";
		else if (nd.rank < 0) os << " bgcolor=\"yellow\"";
		os << ">" << std::abs(nd.rank) << "</TD>";
		// Print candidate
		os << "<TD colspan=\"4\">" << format(nd.cand) << "</TD>";
		// os << "<TD>" << graph[i].candidate().label() << "</TD>";
		os << "</TR>";
		if (body_size == bot_size) {
			// Print seed example if this is the bottom clause
			os << "<TR><TD colspan=\"5\">" << seed() << "</TD></TR>";
		}
		os << "</TABLE> >];\n"; // close label
		// Add edges to all children
		for (auto c = nd.down.begin(); c != nd.down.end(); ++c) {
			os << "\t\t" << node_id(nd.id()) << " -> " << node_id((*c)->id()) << ";\n";
		}
	}


	template <typename Cand>
	void lattice<Cand>::print_subgraph(ostream& os, const parameters& params, const vector<Mode>* modes) const
	{
		if (empty()) return;

		// cerr << "subgraph: printing nodes...\n";
		os << "\tsubgraph cluster_" << name() << " {\n";
		std::for_each(begin(),end(),[&](const layer_type::value_type& p){
			// Build DOT file
			print_node(os,p.second,params,modes);
		});

		// cerr << "subgraph: ranking...\n";
		// Rank all clauses on the same level
		auto i = begin();
		if (i != end()) {
			auto j = std::next(i);
			int i_bsize = std::count(i->second.sol.mask().begin(),i->second.sol.mask().end(),1);
			for ( ; j != end(); ++i,++j) {
				int j_bsize = std::count(j->second.sol.mask().begin(),j->second.sol.mask().end(),1);
				if (i_bsize == j_bsize) {
					os << "\t\t{rank=same; " << node_id(i->second.id()) << "; " << node_id(j->second.id()) << "}\n";
				}
				i_bsize = j_bsize;
			}
		}
		// cerr << "subgraph: printing bottom...\n";
		// Print bottom clause if we haven't
		const int bsize = std::count( (--end())->second.sol.mask().begin(), (--end())->second.sol.mask().end(), 1);
		if ( bsize != bot_size ) {
			const std::string& bottom_name = "node_B" + name();
			os << "\t\t" << bottom_name << " [shape=none,label=< <TABLE><TR><TD colspan=\"3\"></TD>"
				<< "<TD>" << bot_size << "</TD></TR><TR><TD colspan=\"4\">" << format(bottom()) << "</TD></TR>"
				<< "<TR><TD colspan=\"4\">" << format(seed()) << "</TD></TR></TABLE> >];\n";
			os << "\t\t{ rank=sink; \"" << bottom_name << "\" }\n";
		}
		os << "\t}\n"; // close the subgraph
		// cerr << "subgraph: done\n";
	}



	//============================== Globals ==============================//

	template <typename Cand>
	std::map<std::unique_ptr<Functor>,pair<int,int>> consistency_distribution(const lattice<Cand>& lat, const parameters& params)
	{
		std::map<std::unique_ptr<Functor>,pair<int,int>> freq; // bottom literal -> (consistent,inconsistent)
		for_each(lat.begin(),lat.end(),[&](const lattice<Cand>::value_type& p){
			const auto& cand = p.second.cand;
			// Is this candidate consistent or not?
			const bool con = p.second.sol.is_consistent(params.force_int(parameters::noise));
			// Consistent
			for_each(cand->body_begin(),cand->body_end(),[&](const Functor& l){
				auto at = freq.find(*l);
				if (at == freq.end()) {
					freq.insert(map<std::unique_ptr<Functor>,pair<int,int>>::value_type(
						*l,
						make_pair(con,!con)));
				} else {
					con ? ++at->second.first : ++at->second.second;
				}
			});
		});
		return freq;
	}


	// Frequency of (pos,neg)
	template <typename Cand>
	std::map<pair<int,int>,double> coverage_distribution(const lattice<Cand>& lat)
	{
		std::map<pair<int,int>,double> freq;
		for_each(lat.begin(),lat.end(),[&](const layer_type::value_type& p){
			++freq[make_pair(p.sol.pos(),p.sol.neg())];
		});
		return freq;
	}

	// Average frequency per level
	template <typename Cand>
	std::map<int,pair<double,double>> level_average(const lattice<Cand>& lat)
	{
		std::map<int,pair<double,double>> freq;
		for (auto lb = begin(); lb != end(); ) {
			const auto level = lb->first;
			auto p = lat.equal_range(level);
			pair<double,double> avg;
			for_each(p.first,p.second,[&](const lattice<Cand>::value_type& p){
				avg.first += p.second.pos();
				avg.second += p.second.neg();
			});
			// normalize
			const double rsize = double(std::distance(p.first,p.second));
			avg.first /= rsize;
			avg.second /= rsize;
			freq[level] = avg;
			lb = p.second;
		}
		return freq;
	}

} // namespace lp


//namespace std {
//	template<> 
//	template <typename Sol>
//	struct hash<lp::lattice<Sol>>
//	{
//		size_t operator()(const lattice<Sol>& s) const
//		{
//			size_t n = 0;
//			long long f = 1;
//			for (auto i = s.mask().begin(); i != s.mask().end(); ++i) {
//				n += f * (*i);
//				f *= 2;
//			}
//			return n;
//		}
//	};
//}


#endif

